/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#ifdef HAVE_WINSOCK_H
#include <winsock.h>
#endif
#ifdef HAVE_IO_H   /* for Win */
#include <io.h>
#endif
#ifdef HAVE_PROCESS_H
#include <process.h>
#endif

#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#ifdef   HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif
#ifdef HAVE_MMAN_H
#include <sys/mman.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_NETINET_IN_SYSTM_H
#include <netinet/in_systm.h>
#endif
#ifdef HAVE_NETINET_IP_H
#include <netinet/ip.h>
#endif
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif


#include "udm_common.h"
#include "udm_cache.h"
#include "udm_utils.h"
#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_services.h"
#include "udm_log.h"
#include "udm_env.h"
#include "udm_conf.h"

#ifndef INADDR_NONE
#define INADDR_NONE ((unsigned long) -1)
#endif

/*#define DEBUG_LOGD 1*/
/*#define DEBUG_LOGD_CMD 1*/

#define SPLDIR  "splitter"

#define STATE_CMD 1
#define STATE_WRD 2
#define MAXCLIENT 128

/************************ type defenitions ************************/ 
typedef struct {
	int  fd;	/* Socket descriptor    */
	size_t  nbytes;	/* Bytes in buffer      */
	size_t  rbytes;	/* Required bytes       */
	size_t  mbytes;	/* Allocated bytes      */
	int  state;	/* Client state         */
	UDM_LOGD_CMD last_cmd;
	union {
		char * buf;	/* Buffer for data   */
		UDM_LOGD_CMD * cmd;
		UDM_LOGD_WRD * wrd;
	} data;	
} UDM_LOGD_CL;

#ifdef DEBUG_LOGD_CMD
void printcmd(int id,UDM_LOGD_CMD *c){
	fprintf(stderr,"cmd=%d nwords=%d stamp=%d url_id=%d\n",c->cmd,c->nwords,(int)c->stamp,c->url_id);
}
#endif

static void init_client(UDM_LOGD_CL * client, int n){
	int i;
	for(i=0;i<n;i++){
		client[i].fd=0;
		client[i].state=0;
		client[i].nbytes=0;
		client[i].rbytes=0;
		client[i].mbytes=1;
		client[i].data.buf=(char*)malloc(1);
	}
}

static void deinit_client(UDM_LOGD_CL * client, int n){
	int i;
	for(i=0;i<n;i++){
		free(client[i].data.buf);
	}
}

static struct flock* file_lock(int type, int whence) {
    static struct flock ret ;
    ret.l_type = type ;
    ret.l_start = 0 ;
    ret.l_whence = whence ;
    ret.l_len = 0 ;
    ret.l_pid = getpid() ;
    return &ret ;
}

static void write_lock(int fd) {  /* an exclusive lock on an entire file */

    fcntl(fd, F_SETLKW, file_lock(F_WRLCK, SEEK_SET));
}

static void un_lock(int fd) { /* ulock an entire file */
  fcntl(fd, F_SETLKW, file_lock(F_UNLCK, SEEK_SET));
}

/********************* SIG Handlers ****************/
static int have_signal=0;


static void sighandler(int sign);

static void init_signals(void){
	/* Set up signals handler*/
#if (WIN32|WINNT)
#else
	signal(SIGPIPE,sighandler);
	signal(SIGHUP,sighandler);
#endif
	signal(SIGINT,sighandler);
	signal(SIGTERM,sighandler);
}

static void sighandler(int sign){
	have_signal=sign;
	init_signals();
}

/*************************************************************/

static int client_action(int id,UDM_DB *db,UDM_LOGD_CL * client){

	switch(client->state){
		
		case STATE_CMD:
			client->last_cmd=*(client->data.cmd);
#ifdef DEBUG_LOGD_CMD
			printcmd(id,client->data.cmd);
#endif
			if(client->last_cmd.nwords>0){
				client->state=STATE_WRD;
				client->rbytes=client->last_cmd.nwords*sizeof(UDM_LOGD_WRD);
			}else{
				if(LogdStoreDoc(db,client->last_cmd,client->data.wrd)) exit(1);
				client->state=STATE_CMD;
				client->rbytes=sizeof(UDM_LOGD_CMD);
				if (UdmSend(client->fd, "O", 1, 0) != 1) { exit(1); }
			}
			break;
		case STATE_WRD:
			if(LogdStoreDoc(db,client->last_cmd,client->data.wrd)) exit(1);
			client->state=STATE_CMD;
			client->rbytes=sizeof(UDM_LOGD_CMD);
			if (UdmSend(client->fd, "O", 1, 0) != 1) { exit(1); }
			break;
	}
	return 0;
}

static void usage(void){

	fprintf(stderr,
"\n\
cachelogd from %s-%s-%s\n\
http://search.mnogo.ru (C) 1998-2002, LavTech Corp.\n\
\n\
Usage: cachelogd [OPTIONS]\n\
\n\
Options are:\n\
  -w /path      choose alternative working /var directory\n\
  -p xxx        listen port xxx\n\
  -v n          verbose level, 0-5\n\
  -h,-?         print this help page and exit\n\
\n\
Please post bug reports and suggestions to http://www.mnogosearch.org/bugs/\n",
	PACKAGE,VERSION,UDM_DBTYPE);

	return;
}

static char pid_name[1024];
static char time_pid[1024];

static char *Logd_time_pid_info(UDM_LOGD *logd){
	time_t t=time(NULL);
	struct tm *tim=localtime(&t);
	
	strftime(time_pid,sizeof(time_pid),"%a %d %T",tim);
	sprintf(time_pid+strlen(time_pid)," [%d]",(int)getpid());
	return(time_pid);
}


static void RotateDelLog(UDM_ENV *Conf) {
  UDM_DB *db = (UDM_DB *)Conf->db;
  UDM_LOGD *logd = &db->LOGD;
  write_lock(db->del_fd);
  {
    char del_log_name[1024];
    int split_fd, nbytes;
    sprintf(del_log_name, "%s%s", db->log_dir, "del-split.log");
    if((split_fd = open(del_log_name, O_WRONLY | O_CREAT | UDM_BINARY, 0600)) == -1) {
      sprintf(db->errstr, "Can't open '%s' for writing: %s\n", del_log_name, strerror(errno));
      UdmLog_noagent(Conf, UDM_LOG_ERROR, "%s %s", Logd_time_pid_info(logd), db->errstr);
      return;
    }
    lseek(db->del_fd, (off_t)0, SEEK_SET);
    while((nbytes = read(db->del_fd, del_log_name, 1024)) > 0) {
      write(split_fd, del_log_name, (size_t)nbytes);
    }
    close(split_fd);
    lseek(db->del_fd, (off_t)0, SEEK_SET);
    ftruncate(db->del_fd, (off_t)0);
  }
  un_lock(db->del_fd);
}

static void exitproc(void){
        /*
        RotateDelLog();
	LogdClose(&LOGD);
	*/
	unlink(pid_name);
#if (WIN32|WINNT)
	WSACleanup();
#endif
}

int main(int argc,char **argv){
	int	flags=O_RDWR|O_CREAT|O_APPEND|UDM_BINARY;
	int	perm=UDM_IWRITE;
	struct	sockaddr_in server_addr;
	struct	sockaddr_in his_addr;
	struct	in_addr bind_address;
	char	var_dir[1024]=UDM_VAR_DIR UDMSLASHSTR;
	int	on=1,i;
	int	ctl_sock, sel;
	struct timeval tval;
	int	pid_fd,ch,port=UDM_LOGD_PORT;
	char	pidbuf[1024];
	fd_set	mask;
	UDM_LOGD_CL log_client[MAXCLIENT];
	UDM_DB	*db=UdmDBInit(NULL);
	UDM_ENV * Conf = NULL;
	int log2stderr = 1;
	const char * config_name = UDM_CONF_DIR "/indexer.conf";
	
	Conf = UdmEnvInit(NULL);
		
	UdmLoadConfig(Conf, config_name, 0, 0);
	UdmDBFree(Conf->db);
	Conf->db = db;
		
	UdmOpenLog("cachelogd", Conf, log2stderr);

	while ((ch = getopt(argc, argv, "v:w:p:?")) != -1){
		switch (ch) {
			case 'p':
				port=atoi(optarg);
				break;
		        case 'v': UdmSetLogLevel(Conf, atoi(optarg)); break;
			case 'w':
				strcpy(var_dir,optarg);
				break;
			case 'h':
			case '?':
			default:
				usage();
				return 1;
				break;
		}
	}
	argc -= optind;argv += optind;
	
	if(argc>0){
		usage();
		goto err1;
	}
	
	/* Check that another instance isn't running and create PID file */
	sprintf(pid_name,"%s%s",var_dir,"cachelogd.pid");
	pid_fd=open(pid_name,flags,perm);
	if(pid_fd<0){
		UdmLog_noagent(Conf, UDM_LOG_ERROR, "%s Can't create '%s': %s", Logd_time_pid_info(&db->LOGD), pid_name, strerror(errno));
		if(errno==EEXIST){
			UdmLog_noagent(Conf, UDM_LOG_ERROR, "It seems that another cachelogd is already running!\n
Remove '%s' if it is not true.", pid_name);
		}
		goto err1;
	}
	sprintf(pidbuf,"%d\n",(int)getpid());
	write(pid_fd,&pidbuf,strlen(pidbuf));
	close(pid_fd);
	
#if (WIN32|WINNT)
	{
		WSADATA wsaData;

		if(WSAStartup(0x101,&wsaData)!=0){
			UdmLog_noagent(Conf, UDM_LOG_ERROR, "%s WSAStartup() error %d", Logd_time_pid_info(&LOGD), WSAGetLastError);
			goto err1;
		}
	}
#endif
	/* Initialize variables */
	
	if(UDM_OK!=LogdInit(db,var_dir)){
		UdmLog_noagent(Conf, UDM_LOG_ERROR, "Error: %s", db->errstr);
		goto err1;
	}
	
	atexit(&exitproc);
	init_client(log_client,MAXCLIENT);
	init_signals();
	
	UdmLog_noagent(Conf, UDM_LOG_INFO, "%s Started on port %d. Accepting %d connections",
		       Logd_time_pid_info(&db->LOGD), port, sizeof(mask));
	
	ctl_sock = socket(AF_INET, SOCK_STREAM, 0);
	if (ctl_sock < 0) {
		UdmLog_noagent(Conf, UDM_LOG_ERROR , "%s socket() error %d", Logd_time_pid_info(&db->LOGD), errno);
		unlink(pid_name);
		goto err1;
	}
	if (setsockopt(ctl_sock, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on)) != 0){
		UdmLog_noagent(Conf, UDM_LOG_ERROR, "%s setsockopt() error %d", Logd_time_pid_info(&db->LOGD), errno);
		unlink(pid_name);
		goto err1;
	}
	
	
	/* Prepare to start TCP server */
	
	bind_address.s_addr 	= htonl(INADDR_ANY);
	server_addr.sin_family	= AF_INET;
	server_addr.sin_addr	= bind_address;
	server_addr.sin_port	= ntohs(port);
	
	if (bind(ctl_sock, (struct sockaddr *)&server_addr, sizeof(server_addr))) {
		UdmLog_noagent(Conf, UDM_LOG_ERROR, "%s bind() error %d %s", Logd_time_pid_info(&db->LOGD), errno, strerror(errno));
		unlink(pid_name);
		goto err1;
	}
	if (listen(ctl_sock, 32) < 0) {
		UdmLog_noagent(Conf, UDM_LOG_ERROR, "%s listen() error %d %s", Logd_time_pid_info(&db->LOGD), errno, strerror(errno));
		unlink(pid_name);
		goto err1;
	}

	FD_ZERO(&mask);
	FD_SET(ctl_sock,&mask);

	
	while (1) {
		fd_set msk;
		
		tval.tv_sec  		= 300;
		tval.tv_usec 		= 0;
		msk=mask;
		sel=select(MAXCLIENT, &msk, 0, 0, &tval);
		
		if(have_signal){
			UdmLog_noagent(Conf, UDM_LOG_INFO, "%s Signal %d arrived.", Logd_time_pid_info(&db->LOGD), have_signal);
			switch (have_signal){
				case SIGHUP: 
					RotateDelLog(Conf);
					UdmLog_noagent(Conf, UDM_LOG_INFO, "%s Flushing all buffers... ", Logd_time_pid_info(&db->LOGD));
					if(UDM_OK!=LogdSaveAllBufs(db)){
						UdmLog_noagent(Conf, UDM_LOG_ERROR, "%s Error: %s", 
							       Logd_time_pid_info(&db->LOGD), db->errstr);
						UdmLog_noagent(Conf, UDM_LOG_ERROR,"%s Shutdown", Logd_time_pid_info(&db->LOGD));
					}
					UdmLog_noagent(Conf, UDM_LOG_INFO, "Done");
					break;
				case SIGPIPE: 
					UdmLog_noagent(Conf, UDM_LOG_ERROR, "%s FATAL: This should not ever happen! Broken pipe!",
						       Logd_time_pid_info(&db->LOGD));
					break;
				case SIGINT: 
					RotateDelLog(Conf);
					LogdSaveAllBufs(db);
				default:
					UdmLog_noagent(Conf, UDM_LOG_ERROR, "%s Shutdown", Logd_time_pid_info(&db->LOGD));
					goto err0;
			}
			have_signal=0;
			continue;
		}

		
		if(sel==0){
			/* Time limit expired */
 			continue;
		}
		if(sel<1){
			/* FIXME: add errno checking */
			continue;
		}
		
		/* Check whether new client has connected */
		if(FD_ISSET(ctl_sock,&msk)){
			int cl, fd;
			socklen_t addrlen;
			
			addrlen = sizeof(his_addr);
			fd = accept(ctl_sock, (struct sockaddr *)&his_addr, &addrlen);
			if (fd <= 0) {
			  continue;
			}
			for(cl=0;cl<MAXCLIENT;cl++){
				if(log_client[cl].fd==0){
				        UdmLog_noagent(Conf, UDM_LOG_INFO, "%s Client #%d connected", Logd_time_pid_info(&db->LOGD),cl);
					log_client[cl].fd = fd;
					FD_SET(log_client[cl].fd,&mask);
					log_client[cl].state=STATE_CMD;
					log_client[cl].rbytes=sizeof(UDM_LOGD_CMD);
					break;
				}
			}
		}
		
		
		/* Process all ready clients */
		for(i=0;i<MAXCLIENT;i++){
			if(FD_ISSET(log_client[i].fd,&msk)){
				UDM_LOGD_CL * cl;
				int lbytes;
				
				cl=&log_client[i];
#ifdef DEBUG_LOGD
				UdmLog_noagent(Conf, UDM_LOG_DEBUG, "[%d] try read %d r=%d n=%d",
					       i, cl->rbytes-cl->nbytes, cl->rbytes, cl->nbytes);
#endif

				if((cl->mbytes)!=(cl->rbytes)){
					cl->data.buf=(char*)realloc(cl->data.buf,cl->rbytes);
					cl->mbytes=cl->rbytes;
				}

				lbytes = UdmRecvall(cl->fd,&cl->data.buf[cl->nbytes],cl->rbytes-cl->nbytes);
#ifdef DEBUG_LOGD
				UdmLog_noagent(Conf, UDM_LOG_DEBUG, "[%d] read %d bytes n=%d", i, lbytes, cl->nbytes);
#endif				
				if(lbytes){
					cl->nbytes+=lbytes;
#ifdef DEBUG_LOGD
					UdmLog_noagent(Conf, UDM_LOG_DEBUG, "[%d] AAA n[i]=%d r[i]=%d", i, cl->nbytes, cl->rbytes);
#endif					
					if(cl->nbytes==cl->rbytes){
#ifdef DEBUG_LOGD
						UdmLog_noagent(Conf, UDM_LOG_DEBUG, "[%d] struct read %d bytes", i, cl->nbytes);
#endif
						client_action(i,db,cl);
						cl->nbytes=0;
					}
				}else{
					UdmLog_noagent(Conf, UDM_LOG_INFO, "%s Client #%d left", Logd_time_pid_info(&db->LOGD), i);
					FD_CLR(cl->fd,&mask);
					closesocket(cl->fd);
					UDM_FREE(cl->data.buf);
					init_client(cl,1);
				}
			}
		}
	}
err0:
	UdmEnvFree(Conf);
	deinit_client(log_client,MAXCLIENT);
	exit(0);
err1:
	UdmEnvFree(Conf);
	deinit_client(log_client,MAXCLIENT);
	exit(1);

}
